export type WithdrawStatus = "pending" | "approved" | "cancelled"

export type WithdrawUserType = "agent" | "hotel"

export interface WithdrawRequest {
  id: string
  userId: string
  userType: WithdrawUserType
  name: string
  email: string
  phone: string
  walletBalance: number
  withdrawAmount: number
  status: WithdrawStatus
  requestedAt: string
  processedAt?: string
  referenceNumber?: string
  cancelReason?: string
  bankDetails: {
    accountHolder: string
    accountNumber: string
    bankName: string
    ifscCode: string
  }
}

export interface WithdrawalStats {
  totalPending: number
  totalPendingAmount: number
  totalApproved: number
  totalApprovedAmount: number
  totalCancelled: number
  totalCancelledAmount: number
}

// In-memory withdrawal data store
class WithdrawalDataStore {
  private withdrawals: Map<string, WithdrawRequest> = new Map()

  constructor() {
    this.initializeSampleData()
  }

  private initializeSampleData() {
    const sampleWithdrawals: WithdrawRequest[] = [
      // Agent withdrawals
      {
        id: "wd_001",
        userId: "agent_001",
        userType: "agent",
        name: "Rajesh Travel Services",
        email: "rajesh.travels@email.com",
        phone: "+91 9876543210",
        walletBalance: 125000,
        withdrawAmount: 50000,
        status: "pending",
        requestedAt: "2024-12-05T10:30:00Z",
        bankDetails: {
          accountHolder: "Rajesh Kumar",
          accountNumber: "1234567890123456",
          bankName: "State Bank of India",
          ifscCode: "SBIN0001234",
        },
      },
      {
        id: "wd_002",
        userId: "agent_002",
        userType: "agent",
        name: "Priya Tours & Travels",
        email: "priya.tours@email.com",
        phone: "+91 9876543211",
        walletBalance: 78000,
        withdrawAmount: 25000,
        status: "pending",
        requestedAt: "2024-12-04T14:15:00Z",
        bankDetails: {
          accountHolder: "Priya Sharma",
          accountNumber: "9876543210987654",
          bankName: "HDFC Bank",
          ifscCode: "HDFC0004567",
        },
      },
      {
        id: "wd_003",
        userId: "agent_003",
        userType: "agent",
        name: "Amit Pilgrimage Services",
        email: "amit.pilgrimage@email.com",
        phone: "+91 9876543212",
        walletBalance: 215000,
        withdrawAmount: 100000,
        status: "approved",
        requestedAt: "2024-12-01T09:00:00Z",
        processedAt: "2024-12-02T11:30:00Z",
        referenceNumber: "TXN2024120200001",
        bankDetails: {
          accountHolder: "Amit Patel",
          accountNumber: "5678901234567890",
          bankName: "ICICI Bank",
          ifscCode: "ICIC0007890",
        },
      },
      {
        id: "wd_004",
        userId: "agent_004",
        userType: "agent",
        name: "Sunita Yatra Agency",
        email: "sunita.yatra@email.com",
        phone: "+91 9876543213",
        walletBalance: 32000,
        withdrawAmount: 15000,
        status: "cancelled",
        requestedAt: "2024-11-28T16:45:00Z",
        processedAt: "2024-11-29T10:00:00Z",
        cancelReason: "Invalid bank details provided. Account number does not match account holder name.",
        bankDetails: {
          accountHolder: "Sunita Verma",
          accountNumber: "1111222233334444",
          bankName: "Axis Bank",
          ifscCode: "UTIB0001234",
        },
      },
      {
        id: "wd_005",
        userId: "agent_005",
        userType: "agent",
        name: "Vikram Holy Tours",
        email: "vikram.holy@email.com",
        phone: "+91 9876543214",
        walletBalance: 92000,
        withdrawAmount: 40000,
        status: "pending",
        requestedAt: "2024-12-06T08:20:00Z",
        bankDetails: {
          accountHolder: "Vikram Singh",
          accountNumber: "4444555566667777",
          bankName: "Punjab National Bank",
          ifscCode: "PUNB0012345",
        },
      },
      // Hotel withdrawals
      {
        id: "wd_101",
        userId: "hotel_001",
        userType: "hotel",
        name: "Divine Stay Resort",
        email: "divine.stay@email.com",
        phone: "+91 9988776655",
        walletBalance: 450000,
        withdrawAmount: 200000,
        status: "pending",
        requestedAt: "2024-12-05T12:00:00Z",
        bankDetails: {
          accountHolder: "Divine Stay Pvt Ltd",
          accountNumber: "8888999900001111",
          bankName: "Kotak Mahindra Bank",
          ifscCode: "KKBK0001234",
        },
      },
      {
        id: "wd_102",
        userId: "hotel_002",
        userType: "hotel",
        name: "Pilgrim Palace Hotel",
        email: "pilgrim.palace@email.com",
        phone: "+91 9988776656",
        walletBalance: 320000,
        withdrawAmount: 150000,
        status: "pending",
        requestedAt: "2024-12-04T09:30:00Z",
        bankDetails: {
          accountHolder: "Pilgrim Palace Hotels LLP",
          accountNumber: "2222333344445555",
          bankName: "Yes Bank",
          ifscCode: "YESB0005678",
        },
      },
      {
        id: "wd_103",
        userId: "hotel_003",
        userType: "hotel",
        name: "Spiritual Inn",
        email: "spiritual.inn@email.com",
        phone: "+91 9988776657",
        walletBalance: 180000,
        withdrawAmount: 80000,
        status: "approved",
        requestedAt: "2024-11-30T15:45:00Z",
        processedAt: "2024-12-01T10:15:00Z",
        referenceNumber: "TXN2024120100002",
        bankDetails: {
          accountHolder: "Spiritual Inn Hotels",
          accountNumber: "6666777788889999",
          bankName: "Bank of Baroda",
          ifscCode: "BARB0009876",
        },
      },
      {
        id: "wd_104",
        userId: "hotel_004",
        userType: "hotel",
        name: "Sacred Stay Lodge",
        email: "sacred.stay@email.com",
        phone: "+91 9988776658",
        walletBalance: 95000,
        withdrawAmount: 50000,
        status: "cancelled",
        requestedAt: "2024-11-25T11:20:00Z",
        processedAt: "2024-11-26T14:30:00Z",
        cancelReason: "Withdrawal request exceeds maximum limit for unverified hotels.",
        bankDetails: {
          accountHolder: "Sacred Stay Lodge",
          accountNumber: "0000111122223333",
          bankName: "IndusInd Bank",
          ifscCode: "INDB0001234",
        },
      },
      {
        id: "wd_105",
        userId: "hotel_005",
        userType: "hotel",
        name: "Holy Heritage Hotel",
        email: "holy.heritage@email.com",
        phone: "+91 9988776659",
        walletBalance: 275000,
        withdrawAmount: 120000,
        status: "approved",
        requestedAt: "2024-11-28T08:00:00Z",
        processedAt: "2024-11-29T09:45:00Z",
        referenceNumber: "TXN2024112900003",
        bankDetails: {
          accountHolder: "Holy Heritage Hotels Pvt Ltd",
          accountNumber: "5555666677778888",
          bankName: "Federal Bank",
          ifscCode: "FDRL0005678",
        },
      },
      {
        id: "wd_106",
        userId: "hotel_006",
        userType: "hotel",
        name: "Moksha Retreat",
        email: "moksha.retreat@email.com",
        phone: "+91 9988776660",
        walletBalance: 540000,
        withdrawAmount: 300000,
        status: "pending",
        requestedAt: "2024-12-06T07:15:00Z",
        bankDetails: {
          accountHolder: "Moksha Retreat LLP",
          accountNumber: "9999888877776666",
          bankName: "IDFC First Bank",
          ifscCode: "IDFB0001234",
        },
      },
    ]

    sampleWithdrawals.forEach((withdrawal) => {
      this.withdrawals.set(withdrawal.id, withdrawal)
    })
  }

  getWithdrawalsByType(userType: WithdrawUserType): WithdrawRequest[] {
    return Array.from(this.withdrawals.values())
      .filter((w) => w.userType === userType)
      .sort((a, b) => new Date(b.requestedAt).getTime() - new Date(a.requestedAt).getTime())
  }

  getWithdrawalsByTypeAndStatus(userType: WithdrawUserType, status: WithdrawStatus): WithdrawRequest[] {
    return this.getWithdrawalsByType(userType).filter((w) => w.status === status)
  }

  getWithdrawalById(id: string): WithdrawRequest | undefined {
    return this.withdrawals.get(id)
  }

  approveWithdrawal(id: string, amount: number, referenceNumber: string): WithdrawRequest | undefined {
    const withdrawal = this.withdrawals.get(id)
    if (!withdrawal || withdrawal.status !== "pending") return undefined

    const updatedWithdrawal: WithdrawRequest = {
      ...withdrawal,
      status: "approved",
      withdrawAmount: amount,
      referenceNumber,
      processedAt: new Date().toISOString(),
      walletBalance: withdrawal.walletBalance - amount,
    }
    this.withdrawals.set(id, updatedWithdrawal)
    return updatedWithdrawal
  }

  cancelWithdrawal(id: string, reason: string): WithdrawRequest | undefined {
    const withdrawal = this.withdrawals.get(id)
    if (!withdrawal || withdrawal.status !== "pending") return undefined

    const updatedWithdrawal: WithdrawRequest = {
      ...withdrawal,
      status: "cancelled",
      cancelReason: reason,
      processedAt: new Date().toISOString(),
    }
    this.withdrawals.set(id, updatedWithdrawal)
    return updatedWithdrawal
  }

  getWithdrawalStats(userType: WithdrawUserType): WithdrawalStats {
    const withdrawals = this.getWithdrawalsByType(userType)

    const pending = withdrawals.filter((w) => w.status === "pending")
    const approved = withdrawals.filter((w) => w.status === "approved")
    const cancelled = withdrawals.filter((w) => w.status === "cancelled")

    return {
      totalPending: pending.length,
      totalPendingAmount: pending.reduce((sum, w) => sum + w.withdrawAmount, 0),
      totalApproved: approved.length,
      totalApprovedAmount: approved.reduce((sum, w) => sum + w.withdrawAmount, 0),
      totalCancelled: cancelled.length,
      totalCancelledAmount: cancelled.reduce((sum, w) => sum + w.withdrawAmount, 0),
    }
  }
}

export const withdrawalDataStore = new WithdrawalDataStore()
