export type CustomerStatus = "active" | "blocked"

export interface CustomerBooking {
  id: string
  packageName: string
  destination: string
  travelDate: string
  amount: number
  status: "completed" | "upcoming" | "cancelled"
}

export interface Customer {
  id: string
  firstName: string
  lastName: string
  email: string
  phone: string
  status: CustomerStatus
  totalBookings: number
  totalSpent: number
  joinedAt: string
  lastActive: string
  blockReason?: string
  blockedAt?: string
  bookings: CustomerBooking[]
}

// In-memory customer data store
class CustomerDataStore {
  private customers: Map<string, Customer> = new Map()

  constructor() {
    // Initialize with sample customers
    this.initializeSampleData()
  }

  private initializeSampleData() {
    const sampleCustomers: Customer[] = [
      {
        id: "cust_001",
        firstName: "Rajesh",
        lastName: "Kumar",
        email: "rajesh.kumar@email.com",
        phone: "+91 9876543210",
        status: "active",
        totalBookings: 5,
        totalSpent: 125000,
        joinedAt: "2024-01-15",
        lastActive: "2024-12-01",
        bookings: [
          {
            id: "book_001",
            packageName: "Char Dham Yatra",
            destination: "Uttarakhand",
            travelDate: "2024-05-15",
            amount: 35000,
            status: "completed",
          },
          {
            id: "book_002",
            packageName: "Varanasi Spiritual Tour",
            destination: "Varanasi",
            travelDate: "2024-08-20",
            amount: 15000,
            status: "completed",
          },
          {
            id: "book_003",
            packageName: "Tirupati Darshan",
            destination: "Tirupati",
            travelDate: "2025-01-10",
            amount: 25000,
            status: "upcoming",
          },
        ],
      },
      {
        id: "cust_002",
        firstName: "Priya",
        lastName: "Sharma",
        email: "priya.sharma@email.com",
        phone: "+91 9876543211",
        status: "active",
        totalBookings: 3,
        totalSpent: 78000,
        joinedAt: "2024-03-20",
        lastActive: "2024-11-28",
        bookings: [
          {
            id: "book_004",
            packageName: "Amarnath Yatra",
            destination: "Jammu & Kashmir",
            travelDate: "2024-07-10",
            amount: 45000,
            status: "completed",
          },
          {
            id: "book_005",
            packageName: "Shirdi Tour",
            destination: "Maharashtra",
            travelDate: "2024-09-15",
            amount: 12000,
            status: "completed",
          },
        ],
      },
      {
        id: "cust_003",
        firstName: "Amit",
        lastName: "Patel",
        email: "amit.patel@email.com",
        phone: "+91 9876543212",
        status: "active",
        totalBookings: 8,
        totalSpent: 215000,
        joinedAt: "2023-11-10",
        lastActive: "2024-12-02",
        bookings: [
          {
            id: "book_006",
            packageName: "Kailash Mansarovar",
            destination: "Tibet",
            travelDate: "2024-06-01",
            amount: 85000,
            status: "completed",
          },
        ],
      },
      {
        id: "cust_004",
        firstName: "Sunita",
        lastName: "Verma",
        email: "sunita.verma@email.com",
        phone: "+91 9876543213",
        status: "blocked",
        totalBookings: 2,
        totalSpent: 32000,
        joinedAt: "2024-02-28",
        lastActive: "2024-08-15",
        blockReason: "Multiple payment failures and fraudulent activity",
        blockedAt: "2024-08-20",
        bookings: [
          {
            id: "book_007",
            packageName: "Rishikesh Retreat",
            destination: "Uttarakhand",
            travelDate: "2024-04-10",
            amount: 18000,
            status: "cancelled",
          },
        ],
      },
      {
        id: "cust_005",
        firstName: "Vikram",
        lastName: "Singh",
        email: "vikram.singh@email.com",
        phone: "+91 9876543214",
        status: "active",
        totalBookings: 4,
        totalSpent: 92000,
        joinedAt: "2024-04-05",
        lastActive: "2024-11-30",
        bookings: [],
      },
      {
        id: "cust_006",
        firstName: "Meera",
        lastName: "Joshi",
        email: "meera.joshi@email.com",
        phone: "+91 9876543215",
        status: "blocked",
        totalBookings: 1,
        totalSpent: 15000,
        joinedAt: "2024-06-12",
        lastActive: "2024-07-20",
        blockReason: "Violation of terms of service - abusive behavior towards staff",
        blockedAt: "2024-07-25",
        bookings: [],
      },
      {
        id: "cust_007",
        firstName: "Arjun",
        lastName: "Reddy",
        email: "arjun.reddy@email.com",
        phone: "+91 9876543216",
        status: "active",
        totalBookings: 6,
        totalSpent: 145000,
        joinedAt: "2023-09-18",
        lastActive: "2024-12-01",
        bookings: [],
      },
      {
        id: "cust_008",
        firstName: "Kavita",
        lastName: "Nair",
        email: "kavita.nair@email.com",
        phone: "+91 9876543217",
        status: "active",
        totalBookings: 2,
        totalSpent: 48000,
        joinedAt: "2024-07-22",
        lastActive: "2024-11-25",
        bookings: [],
      },
    ]

    sampleCustomers.forEach((customer) => {
      this.customers.set(customer.id, customer)
    })
  }

  getAllCustomers(): Customer[] {
    return Array.from(this.customers.values())
  }

  getActiveCustomers(): Customer[] {
    return this.getAllCustomers().filter((c) => c.status === "active")
  }

  getBlockedCustomers(): Customer[] {
    return this.getAllCustomers().filter((c) => c.status === "blocked")
  }

  getCustomerById(id: string): Customer | undefined {
    return this.customers.get(id)
  }

  blockCustomer(id: string, reason: string): Customer | undefined {
    const customer = this.customers.get(id)
    if (!customer) return undefined

    const updatedCustomer: Customer = {
      ...customer,
      status: "blocked",
      blockReason: reason,
      blockedAt: new Date().toISOString().split("T")[0],
    }
    this.customers.set(id, updatedCustomer)
    return updatedCustomer
  }

  unblockCustomer(id: string): Customer | undefined {
    const customer = this.customers.get(id)
    if (!customer) return undefined

    const updatedCustomer: Customer = {
      ...customer,
      status: "active",
      blockReason: undefined,
      blockedAt: undefined,
    }
    this.customers.set(id, updatedCustomer)
    return updatedCustomer
  }
}

export const customerDataStore = new CustomerDataStore()
