"use server"

import { userDataStore } from "@/lib/data/users"
import { createResetFlowState, markOTPVerified, clearResetFlowState, getResetFlowState } from "@/lib/auth/session"
import { sanitizeInput } from "@/lib/security/sanitize"

interface ActionResult {
  success: boolean
  error?: string
}

interface OTPResult extends ActionResult {
  otp?: string
}

/**
 * Server action to initiate password reset flow
 */
export async function initiatePasswordResetAction(email: string): Promise<ActionResult> {
  try {
    const sanitizedEmail = sanitizeInput(email)
    const user = userDataStore.getUserByEmail(sanitizedEmail)

    if (!user) {
      // Don't reveal if email exists for security
      return { success: false, error: "If an account exists with this email, a verification code will be sent" }
    }

    // Create OTP
    userDataStore.createOTP(sanitizedEmail, "password-reset")

    // Create server-side reset flow state
    await createResetFlowState(sanitizedEmail)

    // In production, send OTP via email here
    // For demo purposes, OTP is stored and would be sent via email service

    return { success: true }
  } catch {
    return { success: false, error: "Failed to initiate password reset" }
  }
}

/**
 * Server action to verify OTP
 */
export async function verifyOTPAction(email: string, otp: string): Promise<ActionResult> {
  try {
    const state = await getResetFlowState()

    if (!state || state.email !== email) {
      return { success: false, error: "Invalid session. Please start over." }
    }

    const isValid = userDataStore.verifyOTP(email, otp, "password-reset")

    if (!isValid) {
      return { success: false, error: "Invalid or expired OTP code" }
    }

    // Mark OTP as verified in server-side state
    await markOTPVerified()

    return { success: true }
  } catch {
    return { success: false, error: "Verification failed" }
  }
}

/**
 * Server action to resend OTP
 */
export async function resendOTPAction(email: string): Promise<ActionResult> {
  try {
    const state = await getResetFlowState()

    if (!state || state.email !== email) {
      return { success: false, error: "Invalid session. Please start over." }
    }

    // Create new OTP
    userDataStore.createOTP(email, "password-reset")

    // In production, send OTP via email here

    return { success: true }
  } catch {
    return { success: false, error: "Failed to resend OTP" }
  }
}

/**
 * Server action to reset password
 */
export async function resetPasswordAction(email: string, newPassword: string): Promise<ActionResult> {
  try {
    const state = await getResetFlowState()

    if (!state || state.email !== email || !state.otpVerified) {
      return { success: false, error: "Invalid session. Please complete the verification process." }
    }

    const user = userDataStore.getUserByEmail(email)

    if (!user) {
      return { success: false, error: "User not found" }
    }

    // Update password
    userDataStore.updateUser(user.id, { password: newPassword })

    // Clear reset flow state
    await clearResetFlowState()

    return { success: true }
  } catch {
    return { success: false, error: "Failed to reset password" }
  }
}

/**
 * Server action to get stored OTP for testing purposes
 * In production, this would not exist - OTP would only be sent via email
 */
export async function getStoredOTPAction(email: string): Promise<OTPResult> {
  try {
    const state = await getResetFlowState()

    if (!state || state.email !== email) {
      return { success: false, error: "Invalid session" }
    }

    // Get the stored OTP from userDataStore
    const otpData = userDataStore.getOTP(email, "password-reset")

    if (!otpData) {
      return { success: false, error: "No OTP found" }
    }

    return { success: true, otp: otpData.code }
  } catch {
    return { success: false, error: "Failed to retrieve OTP" }
  }
}
