"use client"

import { useState, useMemo } from "react"
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table"
import { Input } from "@/components/ui/input"
import { Button } from "@/components/ui/button"
import { Badge } from "@/components/ui/badge"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { Search, ChevronLeft, ChevronRight, ArrowUpDown, ArrowUp, ArrowDown, Eye } from "lucide-react"
import type { HotelBooking } from "@/lib/data/hotel-bookings"
import { ViewBookingDetailsDialog } from "./view-booking-details-dialog"

interface HotelBookingTableProps {
  bookings: HotelBooking[]
}

type SortField = "hotelName" | "customerName" | "roomSelected" | "totalMembers" | "checkInDate" | "totalPrice"
type SortDirection = "asc" | "desc"

const ITEMS_PER_PAGE_OPTIONS = [10, 20, 50, 100]

export function HotelBookingTable({ bookings }: HotelBookingTableProps) {
  const [searchQuery, setSearchQuery] = useState("")
  const [currentPage, setCurrentPage] = useState(1)
  const [itemsPerPage, setItemsPerPage] = useState(10)
  const [sortField, setSortField] = useState<SortField>("checkInDate")
  const [sortDirection, setSortDirection] = useState<SortDirection>("desc")
  const [selectedBooking, setSelectedBooking] = useState<HotelBooking | null>(null)

  const filteredData = useMemo(() => {
    if (!searchQuery.trim()) return bookings
    const query = searchQuery.toLowerCase()
    return bookings.filter(
      (booking) =>
        booking.hotelName.toLowerCase().includes(query) ||
        booking.hotelOwnerName.toLowerCase().includes(query) ||
        booking.customerName.toLowerCase().includes(query) ||
        booking.customerEmail.toLowerCase().includes(query) ||
        booking.roomSelected.toLowerCase().includes(query),
    )
  }, [bookings, searchQuery])

  const sortedData = useMemo(() => {
    return [...filteredData].sort((a, b) => {
      let comparison = 0
      switch (sortField) {
        case "hotelName":
          comparison = a.hotelName.localeCompare(b.hotelName)
          break
        case "customerName":
          comparison = a.customerName.localeCompare(b.customerName)
          break
        case "roomSelected":
          comparison = a.roomSelected.localeCompare(b.roomSelected)
          break
        case "totalMembers":
          comparison = a.totalMembers - b.totalMembers
          break
        case "checkInDate":
          comparison = new Date(a.checkInDate).getTime() - new Date(b.checkInDate).getTime()
          break
        case "totalPrice":
          comparison = a.totalPrice - b.totalPrice
          break
      }
      return sortDirection === "asc" ? comparison : -comparison
    })
  }, [filteredData, sortField, sortDirection])

  const paginatedData = useMemo(() => {
    const startIndex = (currentPage - 1) * itemsPerPage
    return sortedData.slice(startIndex, startIndex + itemsPerPage)
  }, [sortedData, currentPage, itemsPerPage])

  const totalPages = Math.ceil(sortedData.length / itemsPerPage)

  const handleSort = (field: SortField) => {
    if (sortField === field) {
      setSortDirection(sortDirection === "asc" ? "desc" : "asc")
    } else {
      setSortField(field)
      setSortDirection("desc")
    }
  }

  const getSortIcon = (field: SortField) => {
    if (sortField !== field) {
      return <ArrowUpDown className="ml-1 h-3 w-3 text-muted-foreground" />
    }
    return sortDirection === "asc" ? <ArrowUp className="ml-1 h-3 w-3" /> : <ArrowDown className="ml-1 h-3 w-3" />
  }

  const handlePageChange = (page: number) => {
    setCurrentPage(Math.max(1, Math.min(page, totalPages)))
  }

  const handleItemsPerPageChange = (value: string) => {
    setItemsPerPage(Number(value))
    setCurrentPage(1)
  }

  const formatCurrency = (amount: number): string => {
    return new Intl.NumberFormat("en-IN", {
      style: "currency",
      currency: "INR",
      maximumFractionDigits: 0,
    }).format(amount)
  }

  const formatDate = (dateString: string): string => {
    return new Date(dateString).toLocaleDateString("en-IN", {
      day: "2-digit",
      month: "short",
      year: "numeric",
    })
  }

  return (
    <div className="space-y-4">
      <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
        <div className="relative max-w-sm">
          <Search
            className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground"
            aria-hidden="true"
          />
          <Input
            placeholder="Search by hotel, customer, room..."
            value={searchQuery}
            onChange={(e) => {
              setSearchQuery(e.target.value)
              setCurrentPage(1)
            }}
            className="pl-9"
            aria-label="Search bookings"
          />
        </div>
        <div className="flex items-center gap-2">
          <span className="text-sm text-muted-foreground">Show:</span>
          <Select value={String(itemsPerPage)} onValueChange={handleItemsPerPageChange}>
            <SelectTrigger className="w-20" aria-label="Items per page">
              <SelectValue />
            </SelectTrigger>
            <SelectContent>
              {ITEMS_PER_PAGE_OPTIONS.map((option) => (
                <SelectItem key={option} value={String(option)}>
                  {option}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>
      </div>

      <div className="flex items-center justify-between text-sm text-muted-foreground">
        <p>
          Showing {paginatedData.length > 0 ? (currentPage - 1) * itemsPerPage + 1 : 0} to{" "}
          {Math.min(currentPage * itemsPerPage, sortedData.length)} of {sortedData.length} bookings
        </p>
        {searchQuery && (
          <Badge variant="secondary" className="font-normal">
            Filtered from {bookings.length} total
          </Badge>
        )}
      </div>

      <div className="rounded-md border overflow-x-auto">
        <div className="min-w-[900px]">
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead className="w-12 text-center">S.No</TableHead>
                <TableHead className="min-w-[140px]">
                  <Button variant="ghost" size="sm" className="-ml-3 h-8" onClick={() => handleSort("hotelName")}>
                    Hotel Name
                    {getSortIcon("hotelName")}
                  </Button>
                </TableHead>
                <TableHead className="min-w-[120px]">Owner Name</TableHead>
                <TableHead className="min-w-[120px]">
                  <Button variant="ghost" size="sm" className="-ml-3 h-8" onClick={() => handleSort("customerName")}>
                    Customer Name
                    {getSortIcon("customerName")}
                  </Button>
                </TableHead>
                <TableHead className="min-w-[100px]">
                  <Button variant="ghost" size="sm" className="-ml-3 h-8" onClick={() => handleSort("roomSelected")}>
                    Room
                    {getSortIcon("roomSelected")}
                  </Button>
                </TableHead>
                <TableHead className="text-center w-20">
                  <Button variant="ghost" size="sm" className="h-8" onClick={() => handleSort("totalMembers")}>
                    Members
                    {getSortIcon("totalMembers")}
                  </Button>
                </TableHead>
                <TableHead className="text-right min-w-[100px]">
                  <Button variant="ghost" size="sm" className="-mr-3 h-8" onClick={() => handleSort("totalPrice")}>
                    Price
                    {getSortIcon("totalPrice")}
                  </Button>
                </TableHead>
                <TableHead className="text-right min-w-[100px]">Commission</TableHead>
                <TableHead className="text-center w-28">Actions</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {paginatedData.length === 0 ? (
                <TableRow>
                  <TableCell colSpan={9} className="h-24 text-center">
                    <p className="text-muted-foreground">
                      {searchQuery ? `No bookings found matching "${searchQuery}"` : "No bookings found."}
                    </p>
                  </TableCell>
                </TableRow>
              ) : (
                paginatedData.map((booking, index) => (
                  <TableRow key={booking.id}>
                    <TableCell className="text-center font-medium text-muted-foreground">
                      {(currentPage - 1) * itemsPerPage + index + 1}
                    </TableCell>
                    <TableCell className="font-medium">{booking.hotelName}</TableCell>
                    <TableCell className="text-muted-foreground">{booking.hotelOwnerName}</TableCell>
                    <TableCell className="font-medium">{booking.customerName}</TableCell>
                    <TableCell>
                      <Badge variant="outline">{booking.roomSelected}</Badge>
                    </TableCell>
                    <TableCell className="text-center">
                      <Badge variant="secondary">{booking.totalMembers}</Badge>
                    </TableCell>
                    <TableCell className="text-right font-semibold text-emerald-600">
                      {formatCurrency(booking.totalPrice)}
                    </TableCell>
                    <TableCell className="text-right font-medium text-blue-600">
                      {formatCurrency(booking.commission)}
                    </TableCell>
                    <TableCell className="text-center">
                      <Button
                        variant="ghost"
                        size="sm"
                        onClick={() => setSelectedBooking(booking)}
                        aria-label={`View details for ${booking.hotelName}`}
                      >
                        <Eye className="h-4 w-4" />
                        <span className="sr-only">View</span>
                      </Button>
                    </TableCell>
                  </TableRow>
                ))
              )}
            </TableBody>
          </Table>
        </div>
      </div>

      {totalPages > 1 && (
        <div className="flex items-center justify-between">
          <p className="text-sm text-muted-foreground">
            Page {currentPage} of {totalPages}
          </p>
          <div className="flex items-center gap-1">
            <Button
              variant="outline"
              size="icon"
              onClick={() => handlePageChange(currentPage - 1)}
              disabled={currentPage === 1}
              aria-label="Previous page"
            >
              <ChevronLeft className="h-4 w-4" />
            </Button>
            <div className="hidden sm:flex items-center gap-1">
              {Array.from({ length: Math.min(5, totalPages) }, (_, i) => {
                let pageNum: number
                if (totalPages <= 5) {
                  pageNum = i + 1
                } else if (currentPage <= 3) {
                  pageNum = i + 1
                } else if (currentPage >= totalPages - 2) {
                  pageNum = totalPages - 4 + i
                } else {
                  pageNum = currentPage - 2 + i
                }
                return (
                  <Button
                    key={pageNum}
                    variant={currentPage === pageNum ? "default" : "outline"}
                    size="icon"
                    onClick={() => handlePageChange(pageNum)}
                    aria-label={`Go to page ${pageNum}`}
                    aria-current={currentPage === pageNum ? "page" : undefined}
                  >
                    {pageNum}
                  </Button>
                )
              })}
            </div>
            <Button
              variant="outline"
              size="icon"
              onClick={() => handlePageChange(currentPage + 1)}
              disabled={currentPage === totalPages}
              aria-label="Next page"
            >
              <ChevronRight className="h-4 w-4" />
            </Button>
          </div>
        </div>
      )}

      {selectedBooking && (
        <ViewBookingDetailsDialog
          booking={selectedBooking}
          open={!!selectedBooking}
          onOpenChange={(open) => !open && setSelectedBooking(null)}
        />
      )}
    </div>
  )
}
