"use client"

import { useState } from "react"
import { useForm } from "react-hook-form"
import { zodResolver } from "@hookform/resolvers/zod"
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogDescription,
  DialogFooter,
} from "@/components/ui/dialog"
import { Button } from "@/components/ui/button"
import { Textarea } from "@/components/ui/textarea"
import { Label } from "@/components/ui/label"
import { XCircle } from "lucide-react"
import { rejectAgentSchema, type RejectAgentInput } from "@/lib/validations/agent"
import type { Agent } from "@/lib/data/agents"

interface RejectAgentDialogProps {
  agent: Agent | null
  open: boolean
  onOpenChange: (open: boolean) => void
  onConfirm: (reason: string) => void
}

export function RejectAgentDialog({ agent, open, onOpenChange, onConfirm }: RejectAgentDialogProps) {
  const [isSubmitting, setIsSubmitting] = useState(false)

  const {
    register,
    handleSubmit,
    reset,
    formState: { errors },
  } = useForm<RejectAgentInput>({
    resolver: zodResolver(rejectAgentSchema),
    defaultValues: { reason: "" },
  })

  const onSubmit = async (data: RejectAgentInput) => {
    setIsSubmitting(true)
    await new Promise((resolve) => setTimeout(resolve, 300))
    onConfirm(data.reason)
    reset()
    setIsSubmitting(false)
  }

  const handleOpenChange = (newOpen: boolean) => {
    if (!newOpen) reset()
    onOpenChange(newOpen)
  }

  if (!agent) return null

  return (
    <Dialog open={open} onOpenChange={handleOpenChange}>
      <DialogContent className="sm:max-w-md">
        <DialogHeader>
          <div className="flex items-center gap-2">
            <div className="flex h-10 w-10 items-center justify-center rounded-full bg-orange-500/10">
              <XCircle className="h-5 w-5 text-orange-600" aria-hidden="true" />
            </div>
            <div>
              <DialogTitle>Reject Agent Application</DialogTitle>
              <DialogDescription>
                Reject {agent.firstName} {agent.lastName}&apos;s application
              </DialogDescription>
            </div>
          </div>
        </DialogHeader>

        <form onSubmit={handleSubmit(onSubmit)} className="space-y-4">
          <div className="space-y-2">
            <Label htmlFor="reject-reason">
              Reason for rejection <span className="text-destructive">*</span>
            </Label>
            <Textarea
              id="reject-reason"
              placeholder="Please provide a detailed reason for rejecting this application..."
              className="min-h-[120px] resize-none"
              {...register("reason")}
              aria-invalid={errors.reason ? "true" : "false"}
              aria-describedby={errors.reason ? "reject-reason-error" : undefined}
            />
            {errors.reason && (
              <p id="reject-reason-error" className="text-sm text-destructive" role="alert">
                {errors.reason.message}
              </p>
            )}
          </div>

          <DialogFooter className="gap-2 sm:gap-0">
            <Button type="button" variant="outline" onClick={() => handleOpenChange(false)} disabled={isSubmitting}>
              Cancel
            </Button>
            <Button type="submit" variant="destructive" disabled={isSubmitting}>
              {isSubmitting ? "Rejecting..." : "Reject Application"}
            </Button>
          </DialogFooter>
        </form>
      </DialogContent>
    </Dialog>
  )
}
